<?php
// classes/GameEngine.php

// 1. Parent Class: ENTITY
// Contains shared properties and basic initialization
class Entity {
    public $id; 
    public $user_id; 
    public $x; 
    public $y; 
    public $hp; 
    public $maxHp; 
    public $energy; 
    public $hidden = false; 
    public $name;
    public $type;    // defined in children
    public $subtype; // defined in children

    public function __construct($row) {
        $this->id = (int)$row['id'];
        $this->user_id = (int)$row['user_id'];
        $this->x = (int)$row['coord_x'];
        $this->y = (int)$row['coord_y'];
        $this->hp = (int)$row['current_hp'];
        $this->energy = (int)$row['energy'];
        $this->name = $row['nom_vaisseau'] ?? $row['type_ship'];
    }
}

// 2. Child Class: DRONE
// Handles logic specific to Drones (Kamikaze/Scout)
class Drone extends Entity {
    public function __construct($row) {
        parent::__construct($row);
        
        $typeString = strtolower($row['type_ship']);
        $this->type = 'drone';
        
        // Determine subtype based on string
        $this->subtype = (strpos($typeString, 'kamikaze') !== false) ? 'kamikaze' : 'scout';

        // Drones usually have fixed stats if base_hp isn't set
        if (!empty($row['base_hp'])) {
            $this->maxHp = (int)$row['base_hp'];
        } else {
            $this->maxHp = 40; 
        }
    }
}

// 3. Child Class: VAISSEAU
// Handles logic specific to Standard Ships (Chasseur, Croiseur, etc.)
class Vaisseau extends Entity {
    public function __construct($row, $isEnemy) {
        parent::__construct($row);

        // For standard ships, the 'type' tells the frontend if it's mine ('ship') or theirs ('enemy')
        $this->type = ($isEnemy) ? 'enemy' : 'ship';
        $this->subtype = strtolower($row['type_ship']);

        if (!empty($row['base_hp'])) {
            $this->maxHp = (int)$row['base_hp'];
        } else {
            $this->maxHp = $this->hp;
        }
    }
}

// 4. Game Manager
class GameManager {
    private $db;
    private $userId;
    private $opponentId;
    private $gridSize = 10;

    public function __construct($dbLink, $userId) {
        $this->db = $dbLink;
        $this->userId = $userId;
        
        $query = mysqli_query($this->db, "SELECT opponent_id FROM user_profiles WHERE user_id = $this->userId");
        if ($query && mysqli_num_rows($query) > 0) {
            $data = mysqli_fetch_assoc($query);
            $this->opponentId = $data['opponent_id'] ?? 0;
        } else {
            $this->opponentId = 0;
        }
    }

    // --- ACTIONS DE JEU ---
    
    public function move($shipId, $x, $y) {
        if (!$this->isMyTurn()) return ['success' => false, 'msg' => "Ce n'est pas votre tour."];
        if (!$this->isValidPos($x, $y)) return ['success' => false, 'msg' => "Hors zone."];

        $q = mysqli_query($this->db, "SELECT id, coord_x, coord_y, type_ship FROM user_fleets WHERE id = $shipId AND user_id = $this->userId");
        if (mysqli_num_rows($q) === 0) return ['success' => false, 'msg' => "Unité introuvable."];
        $ship = mysqli_fetch_assoc($q);
        
        $dist = abs($ship['coord_x'] - $x) + abs($ship['coord_y'] - $y);
        $maxMove = (strtolower($ship['type_ship']) === 'chasseur') ? 2 : 1;

        if ($dist > $maxMove) return ['success' => false, 'msg' => "Trop loin (Max $maxMove case(s))."];
        if ($this->isOccupied($x, $y)) return ['success' => false, 'msg' => "Case occupée."];

        mysqli_query($this->db, "UPDATE user_fleets SET coord_x = $x, coord_y = $y WHERE id = $shipId");
        return ['success' => true, 'msg' => "Mouvement effectué."];
    }

    public function attack($attackerId, $targetId) {
        if (!$this->isMyTurn()) return ['success' => false, 'msg' => "Pas votre tour."];

        $qAtk = mysqli_query($this->db, "SELECT energy, type_ship FROM user_fleets WHERE id = $attackerId AND user_id = $this->userId");
        $atk = mysqli_fetch_assoc($qAtk);
        if (!$atk || $atk['energy'] < 10) return ['success' => false, 'msg' => "Pas assez d'énergie."];

        $dmg = rand(50, 100); 
        if (strtolower($atk['type_ship']) === 'combat') { $dmg = intval($dmg * 1.25); }
        
        mysqli_query($this->db, "UPDATE user_fleets SET energy = energy - 10 WHERE id = $attackerId");
        
        $actualDmg = $this->applyDamage($targetId, $dmg);
        $res = $this->checkEntityStatus($targetId); 

        return ['success' => true, 'msg' => $res['dead'] ? "Cible détruite !" : "Touché (-$actualDmg PV)."];
    }

    private function applyDamage($targetId, $amount) {
        $qTarget = mysqli_query($this->db, "SELECT type_ship, current_hp FROM user_fleets WHERE id = $targetId");
        $target = mysqli_fetch_assoc($qTarget);
        if (!$target) return 0;

        if (strtolower($target['type_ship']) === 'cuirassé') { $amount = intval($amount * 0.75); }
        $amount = max(1, $amount);

        mysqli_query($this->db, "UPDATE user_fleets SET current_hp = current_hp - $amount WHERE id = $targetId");
        return $amount;
    }

    public function explodeDrone($droneId) {
        if (!$this->isMyTurn()) return ['success' => false, 'msg' => "Pas votre tour."];
        $q = mysqli_query($this->db, "SELECT coord_x, coord_y, type_ship FROM user_fleets WHERE id = $droneId AND user_id = $this->userId");
        $drone = mysqli_fetch_assoc($q);
        if (!$drone || strpos($drone['type_ship'], 'kamikaze') === false) return ['success' => false, 'msg' => "Ce n'est pas un kamikaze."];

        $dx = $drone['coord_x'];
        $dy = $drone['coord_y'];
        $opId = $this->opponentId;

        $sqlTargets = "SELECT id FROM user_fleets WHERE user_id = $opId AND coord_x BETWEEN ($dx - 1) AND ($dx + 1) AND coord_y BETWEEN ($dy - 1) AND ($dy + 1)";
        $res = mysqli_query($this->db, $sqlTargets);
        $hits = 0;
        $dmg = 150;

        while ($target = mysqli_fetch_assoc($res)) {
            $this->applyDamage($target['id'], $dmg);
            $this->checkEntityStatus($target['id']);
            $hits++;
        }
        mysqli_query($this->db, "DELETE FROM user_fleets WHERE id = $droneId");
        return ['success' => true, 'msg' => "BOOM ! $hits cible(s) touchée(s).", 'x' => $dx, 'y' => $dy];
    }

    public function heal($healerId, $targetId) {
        if (!$this->isMyTurn()) return ['success' => false, 'msg' => "Pas votre tour."];
        $qHealer = mysqli_query($this->db, "SELECT uf.coord_x, uf.coord_y, uf.energy, uf.type_ship, v.puissance_tir FROM user_fleets uf JOIN vaisseaux v ON uf.vaisseau_id = v.id WHERE uf.id = $healerId AND uf.user_id = $this->userId");
        $healer = mysqli_fetch_assoc($qHealer);
        
        if (!$healer || $healer['type_ship'] !== 'soigneur') return ['success' => false, 'msg' => "Incapable de soigner."];
        if ($healer['energy'] < 20) return ['success' => false, 'msg' => "Pas assez d'énergie."];

        $qTarget = mysqli_query($this->db, "SELECT uf.coord_x, uf.coord_y, uf.current_hp, v.points_vie as max_hp FROM user_fleets uf JOIN vaisseaux v ON uf.vaisseau_id = v.id WHERE uf.id = $targetId AND uf.user_id = $this->userId");
        $target = mysqli_fetch_assoc($qTarget);

        if (!$target) return ['success' => false, 'msg' => "Cible invalide."];
        if ($target['current_hp'] >= $target['max_hp']) return ['success' => false, 'msg' => "PV déjà au max."];

        $dist = abs($healer['coord_x'] - $target['coord_x']) + abs($healer['coord_y'] - $target['coord_y']);
        if ($dist > 3) return ['success' => false, 'msg' => "Cible trop loin."];

        $healAmount = $healer['puissance_tir'] > 0 ? $healer['puissance_tir'] : 100;
        mysqli_query($this->db, "UPDATE user_fleets SET energy = energy - 20 WHERE id = $healerId");
        $newHp = min($target['current_hp'] + $healAmount, $target['max_hp']);
        mysqli_query($this->db, "UPDATE user_fleets SET current_hp = $newHp WHERE id = $targetId");

        return ['success' => true, 'msg' => "Réparations terminées (+$healAmount PV)."];
    }

    public function spawnDrone($parentId, $type, $x, $y) {
        if (!$this->isMyTurn()) return ['success' => false, 'msg' => "Pas votre tour."];
        if (!$this->isValidPos($x, $y) || $this->isOccupied($x, $y)) return ['success' => false, 'msg' => "Position invalide."];
        
        $qP = mysqli_query($this->db, "SELECT coord_x, coord_y, energy FROM user_fleets WHERE id = $parentId AND user_id = $this->userId");
        $parent = mysqli_fetch_assoc($qP);
        
        $dist = abs($parent['coord_x'] - $x) + abs($parent['coord_y'] - $y);
        if ($dist > 1) return ['success' => false, 'msg' => "Déploiement trop loin."];

        $cost = ($type == 'kamikaze') ? 40 : 20;
        $hp = ($type == 'kamikaze') ? 20 : 40;
        $typeName = ($type == 'kamikaze') ? 'drone_kamikaze' : 'drone_scout';

        if ($parent['energy'] < $cost) return ['success' => false, 'msg' => "Énergie insuffisante."];

        mysqli_query($this->db, "UPDATE user_fleets SET energy = energy - $cost WHERE id = $parentId");
        $sql = "INSERT INTO user_fleets (user_id, vaisseau_id, current_hp, energy, coord_x, coord_y, type_ship) VALUES ($this->userId, 0, $hp, 50, $x, $y, '$typeName')";
        return mysqli_query($this->db, $sql) ? ['success' => true, 'msg' => "Drone déployé."] : ['success' => false, 'msg' => "Erreur SQL"];
    }

    public function endTurn() {
        if (!$this->isMyTurn()) return ['success' => false, 'msg' => "Attendez votre tour."];
        mysqli_query($this->db, "UPDATE user_profiles SET is_my_turn = 0 WHERE user_id = $this->userId");
        mysqli_query($this->db, "UPDATE user_profiles SET is_my_turn = 1 WHERE user_id = $this->opponentId");
        mysqli_query($this->db, "UPDATE user_fleets SET energy = LEAST(energy + 15, 100) WHERE user_id = $this->opponentId");
        return ['success' => true, 'msg' => "Fin du tour."];
    }

    public function surrender() {
        // L'adversaire gagnera automatiquement via getGameState
        mysqli_query($this->db, "DELETE FROM user_fleets WHERE user_id = $this->userId");
        return ['success' => true, 'msg' => "Abandon confirmé."];
    }

    public function quit() {
        mysqli_query($this->db, "DELETE FROM user_fleets WHERE user_id = $this->userId");
        mysqli_query($this->db, "UPDATE user_profiles SET ready = 0, opponent_id = NULL, is_my_turn = 0 WHERE user_id = $this->userId");
        return ['success' => true, 'msg' => "Déconnexion."];
    }

    // --- ETAT ET GESTION VICTOIRE ---

    public function getGameState() {
        if ($this->opponentId == 0) return ['game_over' => true];

        $entities = [];
        $opId = (int)$this->opponentId;

        $sql = "SELECT uf.*, v.points_vie as base_hp, v.nom as nom_vaisseau 
                FROM user_fleets uf
                LEFT JOIN vaisseaux v ON uf.vaisseau_id = v.id
                WHERE (uf.user_id = $this->userId OR uf.user_id = $opId) 
                AND uf.coord_x IS NOT NULL";

        $res = mysqli_query($this->db, $sql);
        $myShips = [];
        $allRows = [];

        if ($res) {
            while ($row = mysqli_fetch_assoc($res)) {
                $allRows[] = $row;
                if ($row['user_id'] == $this->userId) $myShips[] = $row;
            }

            foreach ($allRows as $row) {
                $isEnemy = ($row['user_id'] != $this->userId);
                $typeString = strtolower($row['type_ship']);
                
                // FACTORY LOGIC: Choose Vaisseau or Drone based on type string
                if (strpos($typeString, 'drone') !== false) {
                    $ent = new Drone($row);
                } else {
                    $ent = new Vaisseau($row, $isEnemy);
                }

                // Fog of War Logic (Visibility)
                if ($isEnemy) {
                    $ent->hidden = true;
                    foreach ($myShips as $me) {
                        $range = (strpos(strtolower($me['type_ship']), 'scout') !== false) ? 3 : 2; 
                        $dist = abs($me['coord_x'] - $ent->x) + abs($me['coord_y'] - $ent->y);
                        if ($dist <= $range) { $ent->hidden = false; break; }
                    }
                }
                $entities[] = $ent;
            }
        }

        // Win Condition
        $winStatus = false;
        $nbMe = $this->countShips($this->userId);
        $nbOp = $this->countShips($opId);
        
        if ($nbMe == 0 && $nbOp > 0) $winStatus = 'LOSE';
        
        if ($nbOp == 0 && $nbMe > 0) {
            $winStatus = 'WIN';
            if (!isset($_SESSION['victory_awarded_match_' . $opId])) {
                $this->addVictory($this->userId); 
                $_SESSION['victory_awarded_match_' . $opId] = true;
            }
        }
        
        if ($nbMe == 0 && $nbOp == 0) $winStatus = 'LOSE';

        return [
            'entities' => $entities,
            'turn' => $this->isMyTurn() ? 'player' : 'enemy',
            'win' => $winStatus,
            'game_over' => false
        ];
    }

    // --- HELPERS ---
    
    private function addVictory($userId) {
        $userId = (int)$userId;
        mysqli_query($this->db, "UPDATE user_profiles SET victoires = victoires + 1 WHERE user_id = $userId");
    }

    private function isMyTurn() {
        $q = mysqli_query($this->db, "SELECT is_my_turn FROM user_profiles WHERE user_id = $this->userId");
        $r = mysqli_fetch_assoc($q);
        return ($r && $r['is_my_turn'] == 1);
    }
    private function isOccupied($x, $y) {
        $opId = (int)$this->opponentId;
        $q = mysqli_query($this->db, "SELECT id FROM user_fleets WHERE coord_x=$x AND coord_y=$y AND (user_id=$this->userId OR user_id=$opId)");
        return mysqli_num_rows($q) > 0;
    }
    private function isValidPos($x, $y) { return ($x >= 0 && $x < $this->gridSize && $y >= 0 && $y < $this->gridSize); }
    private function countShips($uid) { $q = mysqli_query($this->db, "SELECT COUNT(*) as c FROM user_fleets WHERE user_id = $uid"); $d = mysqli_fetch_assoc($q); return $d['c']; }
    
    private function checkEntityStatus($id) {
        $q = mysqli_query($this->db, "SELECT current_hp FROM user_fleets WHERE id = $id");
        $d = mysqli_fetch_assoc($q);
        if ($d && $d['current_hp'] <= 0) {
            mysqli_query($this->db, "DELETE FROM user_fleets WHERE id = $id");
            return ['dead' => true];
        }
        return ['dead' => false];
    }
}
?>